const express = require("express");
const userController = require("../controllers/userController");
const authController = require("../controllers/authController");

const router = express.Router();

// ============================================================
// 1. PUBLIC ROUTES
// ============================================================
router.get("/count", userController.getStudentCount);
router.post("/login", authController.login);
router.post("/login-student", authController.loginStudent);
router.post("/google-login", authController.googleLogin);
router.get("/logout", authController.logout);

// OTP Flow
router.post(
  "/signup-student",
  userController.uploadUserPhoto,
  userController.handleBase64Image,
  authController.signupStudent
);

// 🔥 Added Resend Route
router.post("/resend-otp", authController.resendOTP);

router.post("/verify-otp", authController.verifyOTP);

// Password Reset
router.post("/forgotPassword", authController.forgotPassword);
router.patch("/resetPassword/:token", authController.resetPassword);

// ============================================================
// 2. PROTECTED ROUTES
// ============================================================

router.get(
  "/me",
  authController.protect,
  userController.getMe,
  userController.getUser
);

router.patch(
  "/updateMe",
  authController.protect,
  userController.uploadUserPhoto,
  userController.handleBase64Image,
  userController.updateMe
);

router.patch(
  "/updateMyPassword",
  authController.protect,
  authController.updatePassword
);

// ============================================================
// 3. ADMIN ROUTES
// ============================================================

router.get(
  "/",
  authController.protect,
  authController.restrictTo("admin", "super-admin"),
  userController.getAllUsers
);

router.post(
  "/create-admin",
  authController.protect,
  authController.restrictTo("admin", "super-admin"),
  userController.uploadUserPhoto,
  userController.handleBase64Image,
  authController.createAdmin
);

router
  .route("/:id")
  .get(
    authController.protect,
    authController.restrictTo("admin", "super-admin"),
    userController.getUser
  )
  .patch(
    authController.protect,
    authController.restrictTo("admin", "super-admin"),
    userController.uploadUserPhoto,
    userController.handleBase64Image,
    userController.updateUser
  )
  .delete(
    authController.protect,
    authController.restrictTo("admin", "super-admin"),
    userController.deleteUser
  );

module.exports = router;
